#ifndef __TWaveFileFormat__
#define __TWaveFileFormat__

#include <Host/CEndian.hpp>
#include "IAudioFileFormat.hpp"
using Exponent::Audio::IAudioFileFormat;
using Exponent::Host::CEndian;

//	===========================================================================

namespace Exponent
{
	namespace Audio
	{
		/**
		 * @interface TWaveFileFormat TWaveFileFormat.hpp
		 * @brief Specifies the wave file format
		 *
		 * @date 13/04/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @see IAudioFormat
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: TWaveFileFormat.hpp,v 1.9 2007/02/08 21:08:09 paul Exp $
		 */
		template<typename TypeName> class TWaveFileFormat : public IAudioFileFormat<TypeName>
		{
		public:

			/**
			 * Construction
			 */
			TWaveFileFormat()
			{
				m_bitDepth = IAudioFileFormat<TypeName>::e_sixteenBit;
			}

			/**
			 * Destruction
			 */
			virtual ~TWaveFileFormat()
			{
				m_stream.closeStream();
			}

//	===========================================================================

			/**
			 * Open the file
			 * @param mode The mode of opening, read or write
			 * @param filename The name of the file to open
			 * @retval bool True if stream is opened
			 */
			virtual bool openFile(CFileStream::EStreamMode mode, const CSystemString &filename)
			{
				return m_stream.openStream(filename, mode, true);
			}

			/**
			 * Close the file
			 * @retval bool True if closed properly, false otherwise
			 */
			virtual bool closeFile()
			{
				m_stream.closeStream();
				return true;
			}

//	===========================================================================

			/**
			 * Get the extension of the file format, each array entry sould contain one string
			 * @param array The array to fill in
			 */
			virtual void getFileExtension(TStringCountedPointerArray &array) const
			{
				array.addElement(new CString("wav"));
			}

			/**
			 * Check if a file is a valid format of this file format
			 * @param filename The name of the file to load
			 * @retval bool True if the file format matches (checks actual binary data, not just extension
			 */
			virtual bool isValidFormat(const CSystemString &filename) const
			{
				// Create the stream
				CFileStream stream(filename, CFileStream::e_input, true);

				// Check open
				if (!stream.isStreamOpen())
				{
					return false;
				}

				// Read in the header
				typename TWaveFileFormat<TypeName>::SWaveFileFormat diskFormat;
				return readFileHeader(stream, diskFormat);
			}

//	===========================================================================

			/**
			 * Read the header information and store a copy in the format
			 * @param format On return is filled with the values for this audio file
			 * @retval bool True if read the header correctly, false otherwise
			 */
			virtual bool readHeader(typename IAudioFileFormat<TypeName>::SAudioFileFormat &format)
			{
				// Read in the header
				typename TWaveFileFormat<TypeName>::SWaveFileFormat diskFormat;
				if (!readFileHeader(m_stream, diskFormat))
				{
					return false;
				}

				// STore the output information
				format.m_sampleRate		  = diskFormat.m_sampleRate;
				format.m_numberOfChannels = diskFormat.m_numberOfChannels;
				format.m_numberOfSamples  = diskFormat.m_numberOfSamples;
				format.m_bitDepth		  = (unsigned long)diskFormat.m_bitsPerSample;
				m_bitDepth				  = (typename IAudioFileFormat<TypeName>::EBitDepth)format.m_bitDepth;

				// Done :)
				return true;
			}

			/**
			 * Read the entire audio file to the provided audio buffer
			 * @param buffer The data buffer to store te samples in
			 * @retval bool True if read correctly, false otherwise
			 */
			virtual bool readData(TAudioBuffer< TypeName > &buffer)
			{
				// Handle based upon the depth
				switch(m_bitDepth)
				{
					case IAudioFileFormat<TypeName>::e_eightBit:
						{
							// Store the entire size of the buffer to be read in
							const unsigned long size = buffer.getBufferSize();

							// Pointer to the data that we will iterate through
							TypeName *data = buffer.getMutableData();

							// The sample we read in to
							//unsigned char sample = 0;
							unsigned char *sample = new unsigned char[size];

							// Read the audio data
							if (!m_stream.readDataFromStream(sample, size))
							{
								FREE_ARRAY_POINTER(sample);
								return false;
							}

							// Save doing multiple divisions...
							const TypeName inversion = (TypeName)(1.0 / 255.0);

							// For the number of samples, we want to read in and convert to floating point range..
							for (unsigned long i = 0; i < size; i++)
							{
								//m_stream >> sample;
								//*data++ = (TypeName)((((TypeName)sample / (TypeName)255) * 2.0) - 1.0);
								*data++ = (TypeName)((((TypeName)sample[i] * inversion) * 2.0) - 1.0);
							}

							// Delete the buffer we loaded with
							FREE_ARRAY_POINTER(sample);
						}
					break;
					case IAudioFileFormat<TypeName>::e_sixteenBit:
						{
							// Store the entire size of the buffer to be read in
							const unsigned long size = buffer.getBufferSize();

							// Pointer to the data that we will iterate through
							TypeName *data = buffer.getMutableData();

							// The sample we read in to
							short *sample = new short[size];

							// Read the audio data
							if(!m_stream.readShortsFromStream(sample, size))
							{
								FREE_ARRAY_POINTER(sample);
								return false;
							}

							// Save doing multiple divisions...
							const TypeName inversion = (TypeName)(1.0 / 32768.0);

							// For the number of samples, we want to read in and convert to floating point range..
							for (unsigned long i = 0; i < size; i++)
							{
								*data++ = (TypeName)sample[i] * inversion;
							}

							// Delete the sample buffer...
							FREE_ARRAY_POINTER(sample);
						}
					break;
					case IAudioFileFormat<TypeName>::e_twentyFourBit:
						{
							// Store the entire size of the buffer to be read in
							const unsigned long size = buffer.getBufferSize();

							// Pointer to the data that we will iterate through
							TypeName *data = buffer.getMutableData();

							// The sample we read in to
							char streamBuffer[3];

							// For the number of samples, we want to read in and convert to floating point range..
							for (unsigned long i = 0; i < size; i++)
							{
								// Read the 3 bytes from the buffer
								m_stream.readDataFromStream(streamBuffer, 3);

								// Convert sample
								*data++ = (TypeName)((int)CEndian::convertThreeBytesToTwentyFourBitInt(streamBuffer) << 8) / (TypeName)0x7fffffff;
							}
						}
					break;
					default:
						return false;
					break;
				}

				// Sucess!
				return true;
			}

//	===========================================================================

			/**
			 * Write the header information and from the format
			 * @param format The format to write in
			 * @retval bool True if wrote the header correctly, false otherwise
			 */
			virtual bool writeHeader(const typename IAudioFileFormat<TypeName>::SAudioFileFormat &format)
			{
				// Check open
				if (!m_stream.isStreamOpen())
				{
					return false;
				}

				// Store all the information that we need
				const long depth		  = (const long)format.m_bitDepth;
				const long sampleRate	  = (const long)format.m_sampleRate;
				const long numberOfBytes  = format.m_numberOfChannels * depth / 8;
				const long totalSize	  = format.m_numberOfSamples * numberOfBytes + 44 - 8;
				const long bytesPerSample = numberOfBytes * sampleRate;
				const long chunkSize	  = numberOfBytes * format.m_numberOfSamples;
				m_bitDepth				  = (typename IAudioFileFormat<TypeName>::EBitDepth)format.m_bitDepth;

				// Stream the header information
				m_stream << 'R' << 'I' << 'F' << 'F'				// RIFF header
						 << totalSize								// File size
						 << 'W' << 'A' << 'V' << 'E'				// WAVE header
						 << 'f' << 'm' << 't' << ' '				// Format tag
						 << (long)16								// Always 16 for wav
						 << (short)1								// Format 1 PCM
						 << (short)format.m_numberOfChannels		// Number of channels
						 << sampleRate								// Sample rate
						 << (long)bytesPerSample					// bytes per sample
						 << (short)numberOfBytes					// total number of bytes
						 << (short)depth							// bit depth
						 << 'd' << 'a' << 't' << 'a'				// Data tag
						 << chunkSize;								// Size of the data chunk

				// Check we wrote correctly
				return true;
			}

			/**
			 * Write the entire audio file to the provided audio buffer
			 * @param buffer The data buffer to store
			 * @retval bool True if written correctly, false otherwise
			 */
			virtual bool writeData(const TAudioBuffer< TypeName > &buffer)
			{
				switch(m_bitDepth)
				{
					case IAudioFileFormat<TypeName>::e_eightBit:
						{
							const TypeName *data = buffer.getData();
							for (unsigned long i = 0; i < buffer.getNumberOfChannels() * buffer.getNumberOfSamples(); i++)
							{
								//unsigned char sample = (unsigned char)(255.0 * ((data[i] + 1.0) * 0.5));
								m_stream << (unsigned char)(255.0 * ((data[i] + 1.0) * 0.5));//sample;
							}
						}
					break;
					case IAudioFileFormat<TypeName>::e_sixteenBit:
						{
							const TypeName *data = buffer.getData();
							for (unsigned long i = 0; i < buffer.getNumberOfChannels() * buffer.getNumberOfSamples(); i++)
							{
								//short sample = (short)(data[i] * 32768);
								m_stream << (short)(data[i] * 32768);//sample;
							}
						}
					break;
					case IAudioFileFormat<TypeName>::e_twentyFourBit:
						{
							const TypeName *data = buffer.getData();
							char samples[3];
							for (unsigned long i = 0; i < buffer.getNumberOfChannels() * buffer.getNumberOfSamples(); i++)
							{
								CEndian::convertTwentyFourBitIntToThreeBytes((int)(data[i] * (double)0x7fffffff) >> 8, samples);
								m_stream.writeDataToStream(samples, 3);
							}
						}
					break;
					default:
						return false;
					break;
				}

				// Done! :)
				return true;
			}

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * @struct SWaveFileFormat TWaveFileFormat.hpp
			 * @brief Storeage for the file version of the wave
			 */
			struct SWaveFileFormat
			{
				// RIFF chunk
				char m_riffId[4];					/**< 'RIFF' */
				unsigned long m_riffSize;			/**< Size of the file */
				char m_riffFormatId[4];				/**< 'WAVE' */

				// Format chunk
				char m_formatId[4];					/**< 'fmt ' */
				unsigned long m_formatSize;			/**< Size of the format */

				// Wave format chunk
				unsigned short m_formatTag;			/**< format, expects 1 */
				unsigned short m_numberOfChannels;	/**< Number of channels */
				unsigned long m_sampleRate;			/**< Sample rate */
				unsigned long m_bytesPerSecond;		/**< Bytes per second */
				unsigned short m_blockAlignment;	/**< Block alignment */
				unsigned short m_bitsPerSample;		/**< Bits per sample */
				unsigned long m_numberOfSamples;	/**< Number of samples */

				// Data chunk
				char m_dataId[4];					/**< 'data' */
				unsigned long m_dataSize;			/**< Size of the data */
			};

//	===========================================================================

			/**
			 * Is this a valid riff wave
			 * @param stream The stream to read
			 * @param diskFormat The format header to check
			 * @retval bool True if riff wave file format
			 */
			static bool readFileHeader(CFileStream &stream, SWaveFileFormat &diskFormat)
			{
				if (!stream.isStreamOpen())
				{
					return false;
				}

				// Read the riff header
				stream >> diskFormat.m_riffId[0]
					   >> diskFormat.m_riffId[1]
					   >> diskFormat.m_riffId[2]
					   >> diskFormat.m_riffId[3]
					   >> diskFormat.m_riffSize
					   >> diskFormat.m_riffFormatId[0]
					   >> diskFormat.m_riffFormatId[1]
					   >> diskFormat.m_riffFormatId[2]
					   >> diskFormat.m_riffFormatId[3];

				// Is this actually a wave format file?
				if (!isValidRiffWaveChunk(diskFormat))
				{
					return false;
				}

				// ADDED HERE!!

				bool foundTag = false;
				do
				{
					// Read the format
					stream >> diskFormat.m_formatId[0] >> diskFormat.m_formatId[1] >> diskFormat.m_formatId[2] >> diskFormat.m_formatId[3];

					// Read the size of the format
					stream >> diskFormat.m_formatSize;

					// If we found the tag
					foundTag = isValidFormatChunk(diskFormat);

					// Likeyly off the end of the file
					if (stream.hasErrorOccurred())
					{
						return false;
					}

					// If we didnt find the tag
					if (!foundTag)
					{
						stream.advanceStream(diskFormat.m_formatSize);
					}

				}while(!foundTag);

				// Store the format
				stream >> diskFormat.m_formatTag;

				// END ADDED HERE!!

				// Read the format chunk
				//stream >> diskFormat.m_formatId[0] >> diskFormat.m_formatId[1] >> diskFormat.m_formatId[2] >> diskFormat.m_formatId[3];

				// Check that the file is valid for format
				//if (!isValidFormatChunk(diskFormat))
				//{
				//	return false;
				//}

				// Store size and format
				//stream >> diskFormat.m_formatSize >> diskFormat.m_formatTag;

				// Check its PCM encoding
				if (diskFormat.m_formatTag != 1)
				{
					return false;
				}

				// Stream in the format
				stream >> diskFormat.m_numberOfChannels
					   >> diskFormat.m_sampleRate
					   >> diskFormat.m_bytesPerSecond
					   >> diskFormat.m_blockAlignment
					   >> diskFormat.m_bitsPerSample;

				/*
				// REad the data id
				stream >> diskFormat.m_dataId[0]
					   >> diskFormat.m_dataId[1]
					   >> diskFormat.m_dataId[2]
					   >> diskFormat.m_dataId[3];

				// Check that its actually a valid data header
				if (!isValidDataChunk(diskFormat))
				{
					return false;
				}

				// Store the size of the data
				stream >> diskFormat.m_dataSize;
				*/

				foundTag = false;
				
				do
				{
					// REad the data id
					stream >> diskFormat.m_dataId[0] >> diskFormat.m_dataId[1] >> diskFormat.m_dataId[2] >> diskFormat.m_dataId[3] >> diskFormat.m_dataSize;
					
					// If we found the tag
					foundTag = isValidDataChunk(diskFormat);

					// Likeyly off the end of the file
					if (stream.hasErrorOccurred())
					{
						return false;
					}

					// If we didnt find the tag
					if (!foundTag)
					{
						stream.advanceStream(diskFormat.m_dataSize);
					}
				}while(!foundTag);

				// Compute the number of samples
				diskFormat.m_numberOfSamples = diskFormat.m_dataSize / diskFormat.m_blockAlignment;

				// We are done :)
				return true;
			}

			/**
			 * Is this a valid riff wave
			 * @param format The format header to check
			 * @retval bool True if riff wave file format
			 */
			static bool isValidRiffWaveChunk(const SWaveFileFormat &format)
			{
				return (format.m_riffId[0] 		 == 'R' &&
						format.m_riffId[1] 		 == 'I' &&
						format.m_riffId[2] 		 == 'F' &&
						format.m_riffId[3]		 == 'F' &&
						format.m_riffFormatId[0] == 'W' &&
						format.m_riffFormatId[1] == 'A' &&
						format.m_riffFormatId[2] == 'V' &&
						format.m_riffFormatId[3] == 'E');
			}

			/**
			 * Is this a valid format chunk
			 * @param format The format header to check
			 * @retval bool True if the format has a valid format chunk
			 */
			static bool isValidFormatChunk(const SWaveFileFormat &format)
			{
				return (format.m_formatId[0] == 'f' &&
						format.m_formatId[1] == 'm' &&
						format.m_formatId[2] == 't' &&
						format.m_formatId[3] == ' ');
			}

			/**
			 * Is this a valid data chunk
			 * @param format The format header to check
			 * @retval bool True if the format has a valid data chunk
			 */
			static bool isValidDataChunk(const SWaveFileFormat &format)
			{
				return (format.m_dataId[0] == 'd' &&
						format.m_dataId[1] == 'a' &&
						format.m_dataId[2] == 't' &&
						format.m_dataId[3] == 'a');
			}

//	===========================================================================

			CFileStream m_stream;													/**< The file stream that we have */
			typename IAudioFileFormat<TypeName>::EBitDepth m_bitDepth;				/**< Bit depth of files being written */
		};
	}
}
#endif	// End Of TWaveFileFormat.hpp